﻿using System;
using System.Threading;
using System.Windows.Forms;
using Psion.RFID.HF;

namespace HF_Demo
{
    /// <summary>
    /// This form gives access to Operation Mode Register (EEPROM) of RFID reader
    /// It allows to set only the kind of tag that you are focusing on 
    /// </summary>
    public partial class SettingsForm : Form
    {
        private readonly Reader _reader;
        private readonly Config _config;
        private ISOs _iso;

        public SettingsForm(Reader reader)
        {
            InitializeComponent();

            if (!Device.IsWM) mainMenu1.Dispose();

            // get reader instance from parent
            _reader = reader;
            
            // list all the ISO format supported (refer ISOs enumeration)
            foreach (string name in EnumHelper.GetNames(typeof(ISOs)))
                comboBoxTagSelection.Items.Add(name);

            // try to load current config
            try
            {
                _config = Config.Load();
            }
            catch (Exception)
            { // first call, we create the file
                _config = new Config();
                _config.Save();
            }

            // select and display the choice from config file
            comboBoxTagSelection.SelectedItem = _config.TypeOfIso;
        }

        private void ComboBoxTagSelectionSelectedIndexChanged(object sender, EventArgs e)
        {
            string selectedName = comboBoxTagSelection.SelectedItem.ToString();

            // parse to enum
            _iso = (ISOs) Enum.Parse(typeof(ISOs), selectedName, true);
            // update config file
            _config.TypeOfIso = selectedName;

            RefreshDescription();
        }

        private void RefreshDescription()
        {
            switch (_iso)
            {
                case ISOs.ALL:
                    labelDescription.Text = "Supports all the tags:\r\n\tISO 14443 A\r\n\tISO 14443 B\r\n\tISO 15693\r\n\tFelica\r\nGood for tag detection\r\nSlower reading performances";
                    break;
                case ISOs.ISO14443A:
                    labelDescription.Text = "Supports ISO 14443-A tags only\r\nReading timings optimized for this tag";
                    break;
                case ISOs.ISO14443B:
                    labelDescription.Text = "Supports ISO 14443-B tags only\r\nReading timings optimized for this tag";
                    break;
                case ISOs.ISO15693:
                    labelDescription.Text = "Supports ISO 15693 tags only\r\nReading timings optimized for this tag";
                    break;
                case ISOs.Felica:
                    labelDescription.Text = "Supports Felica tags only\r\nReading timings optimized for this tag";
                    break;
            }

            labelDescription.Refresh();
        }

        private void ButtonValidateClick(object sender, EventArgs e)
        {
            ApplySettings();
            Close();
        }

        private void ApplySettings()
        {
            // read current Operation Mode value
            var data = _reader.ReadEeprom(0x0E);
            
            // if new value, apply the setting
            if (data != null && data.Length > 0 && data[0] != (byte)_iso)
            {
                // write new value on EEPROM
                _reader.WriteEeprom(0x0E, (byte) _iso);

                // to apply new setting, the reset is necessary
                Thread.Sleep(50);
                _reader.Reset();
                Thread.Sleep(50);

                // save the change on config file
                _config.Save();
            }
        }
    }
}